# Makefile architecture configuration for the rendering library.
# Copyright 1994-2001 John E. Stone
# All Rights Reserved
#
# $Id: Make-arch,v 1.127 2003/01/22 22:52:13 johns Exp $
#

# Some machines don't have/need ranlib, in their case, use 'touch'
# this should be overridden by arch specific configuration lines below
RANLIB= touch

# MPI path setup, probably shouldn't need to be changed.
MPIINC=$(MPIDIR)/include
MPILIB=$(MPIDIR)/lib

MISCDEFS=$(USEJPEG) $(USEPNG) $(FLT) $(MBOX)
MISCINC=$(JPEGINC) $(PNGINC) $(SPACEBALLINC)
MISCFLAGS=$(MISCDEFS) $(MISCINC)
MISCLIB=$(JPEGLIB) $(PNGLIB) $(SPACEBALLLIB)

default:
	@echo "  Choose one of the architectures specified below."
	@echo "--------------------------------------------------------------"
	@echo "              Parallel Versions        "
	@echo ""
	@echo "             aix-thr - IBM AIX 4.x      POSIX Threads         "
	@echo "             aix-mpi - IBM AIX 4.x (SP) MPI                   "
	@echo "        asci-red-mpi - Intel ASCI Red   MPI                   "
	@echo "            cray-thr - Cray J90,C90...  POSIX Threads         "
	@echo "        cray-t3e-mpi - Cray T3E         MPI                   "
	@echo "  compaq-alphasc-mpi - Lemieux at PSC   MPI                   "
	@echo "        cspi-ppc-mpi - CSPI PowerPC     MSTI-MPI/Pro          "
	@echo "            hpux-thr - HP/UX 11.x       POSIX Threads         "
	@echo "       hpux-ia64-thr - HP-UX IA-64, HP C, POSIX Threads       "
	@echo "         ipsc860-mpi - Intel iPSC/860   MPI                   "
	@echo "           irix5-mpi - SGI IRIX 5.x     MPI                   "
	@echo "           irix6-thr - SGI IRIX 6.5.x   POSIX Threads         "
	@echo "        irix6-64-thr - SGI IRIX 6.5.x   POSIX Threads, 64-bit "
	@echo "   linux-beowulf-mpi - Scyld Linux MPI                        " 
	@echo " linux-alpha-ccc-qsw - Linux Alpha, Compaq C, MPI, QSWnet     "
	@echo "           linux-lam - Linux            MPI (OSC LAM)         "
	@echo "           linux-thr - Linux            POSIX Threads         "
	@echo "      linux-ia64-thr - Linux IA-64, GCC, POSIX Threads        "
	@echo "linux-ia64-thr-sgicc - Linux IA-64, SGI Pro64 Compilers       "
	@echo "          macosx-thr - MacOS-X (aka Darwin), POSIX Threads    "
	@echo "     mercury-ppc-mpi - Mercury PowerPC  MSTI-MPI/Pro          "
	@echo "         paragon-mpi - Intel Paragon    MPI                   "
	@echo "   solaris-ultra-hpc - Sun Solaris 2.8  ClusterTools 4.0 MPI  " 
	@echo "         solaris-mpi - Sun Solaris 2.x  MPI (MPICH)           " 
	@echo "         solaris-lam - Sun Solaris 2.x  MPI (OSC LAM)         " 
	@echo "         solaris-thr - Sun Solaris 2.x  UI/Sun Threads        " 
	@echo "   solaris-ultra-thr - Sun Solaris 2.x  UI/Sun Threads, US-1/2" 
	@echo "  solaris-ultra3-thr - Sun Solaris 2.x  UI/Sun Threads, US-III" 
	@echo "      solaris-64-thr - Sun Solaris 7/8  UI/Sun Threads, 64-bit" 
	@echo "     tru64-alpha-thr - Tru64/OSF1/DUnix POSIX Threads         "
	@echo "     tru64-alpha-omp - Tru64/OSF1/DUnix OpenMP                "
	@echo "    tru64-alpha-ompa - Tru64/OSF1/DUnix OpenMP Assure         "
	@echo ""
	@echo "--------------------------------------------------------------"
	@echo "              Sequential Versions                             "
	@echo ""
	@echo "              aix - IBM AIX 4.x                             "
	@echo "             beos - BeOS, Be Inc's operating system         "
	@echo "              bsd - OpenBSD/FreeBSD/NetBSD                  "
	@echo "             hpux - HP/UX 10.20                             "
	@echo "            irix5 - SGI Irix 5.x                            "
	@echo "            irix6 - SGI Irix 6.x                            "
	@echo "            linux - Linux                                   "
	@echo "    linux-intelcc - Linux, Intel C compilers, P4-optimized  "
	@echo "     linux-athlon - Linux using GCC 3.x Athlon optimizations"
	@echo "         linux-p4 - Linux using GCC 3.x P4 optimizations    "
	@echo "       linux-pgcc - Linux, Portland Group Compilers, Pentium"
	@echo "linux-athlon-pgcc - Linux, Portland Group Compilers, Athlon "
	@echo "        linux-ppc - Linux PowerPC                           "
	@echo "        linux-ps2 - Linux Sony Playstation 2                "
	@echo "      linux-alpha - Linux Alpha                             "
	@echo "  linux-alpha-ccc - Linux Alpha, Compaq Compilers           "
	@echo "       linux-ia64 - Linux IA-64, GCC                        "
	@echo " linux-ia64-sgicc - Linux IA-64, SGI Pro64 Compilers        "
	@echo "           macosx - MacOS-X (aka Darwin, Rhapsody)          "
	@echo "      tru64-alpha - Tru64 Unix, binary can run on AlphaLinux"
	@echo "          solaris - Sun Solaris 2.[3-8]                     "
	@echo "            win32 - Windows 95/NT with Cygnus EGCS/CygWin32 "
	@echo ""
	@echo "--------------------------------------------------------------"
	@echo "              OpenGL Runtime Display Versions                 "
	@echo ""
	@echo "        irix6-thr-ogl - SGI IRIX 6.5.x   POSIX Threads + OpenGL"
	@echo "solaris-ultra-thr-ogl - Sun Solaris 2.x  UI/Sun Threads + OpenGL"
	@echo "solaris-ultra-hpc-ogl - Sun Solaris 2.x  Sun MPI + OpenGL"
	@echo "        linux-thr-ogl - Linux POSIX Threads + OpenGL           "
	@echo "linux-beowulf-mpi-ogl - Scyld Linux with MPI + OpenGL          "
	@echo "  tru64-alpha-thr-ogl - Tru64/OSF1/DUnix POSIX Threads + OpenGL"
	@echo ""
	@echo "--------------------------------------------------------------"
	@echo "Type: 'make arch' to build for an architecture listed above."

##
## Intel ASCI Red (Janus) using MPI.  
## No pthreads on ASCI Red yet.  I didn't bother with the cop() stuff
##
asci-red-mpi:
	$(MAKE) all \
	"ARCH = asci-red-mpi" \
	"CC = cicc" \
	"CFLAGS = -cougar -O4 -Mvect -Knoieee -I$(MPIINC) -Dparagon $(MISCFLAGS) -DMPI" \
	"AR = xar" \
	"ARFLAGS = r" \
	"STRIP = xstrip" \
	"LIBS = -L. -L$(MPILIB)/paragon/ch_nx -nx -ltachyon -lmpi $(MISCLIB) -lm $(MISCLIB) -lmach3"

##
## Cray Parallel Vector Processor Machines Using Threads
##
## Tested on J90s, but should work on almost any of the Cray PVP systems.
## Note: This code is not currently vectorized, and you may be better 
##       off running on a fast workstation, or even better, on a T3E!!!
##
cray-thr:
	@echo "**"
	@echo "** Warning: This software will build on Cray PVP systems;    "
	@echo "**          however the code isn't vectorized.               "
	@echo "**          Measurements on a J90 indicate that the code     "
	@echo "**          only attains 25% of the per-cpu peak vector      "
	@echo "**          performance.  Just FYI for people that don't     "
	@echo "**          want to waste Cray PVP time.                     "
	@echo "**          The efficiency on scalar Cray machines is        "
	@echo "**          excellent (T3E).                                 "
	@echo "**"
	$(MAKE) all \
	"ARCH = cray-thr" \
	"CC = cc" \
	"CFLAGS = -O3 -D_REENTRANT -DTHR $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = -r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon -lpthread $(MISCLIB) -lm"

##
## Cray T3E Using MPI
##
cray-t3e-mpi:
	$(MAKE) all \
	"ARCH = cray-t3e-mpi" \
	"CC = cc" \
	"CFLAGS = -O3 $(MISCFLAGS) -DMPI" \
	"AR = ar" \
	"ARFLAGS = -r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon -lmpi $(MISCLIB) -lm"

# Cray T3E performance tuning via "pat"
cray-t3e-mpi-pat:
	$(MAKE) all \
	"ARCH = cray-t3e-mpi-pat" \
	"CC = cc" \
	"CFLAGS = -O3 $(MISCFLAGS) -DMPI" \
	"AR = ar" \
	"ARFLAGS = -r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon -lmpi $(MISCLIB) -lm -lpat"

# Cray T3E profiling using "apprentice"
cray-t3e-mpi-app:
	$(MAKE) all \
	"ARCH = cray-t3e-mpi-app" \
	"CC = cc" \
	"CFLAGS = -O3 -X1 -happrentice $(MISCFLAGS) -DMPI" \
	"AR = ar" \
	"ARFLAGS = -r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon -lmpi $(MISCLIB) -lm -lapp"

##
## Cray T3E Using MPI
##
compaq-alphasc-mpi:
	$(MAKE) all \
	"ARCH = compaq-alphasc-mpi" \
	"CC = cc" \
	"CFLAGS = -speculate all -std1 -pthread -fast -O4 -arch host -tune host -verbose $(MISCFLAGS) -DMPI -DLP64" \
	"AR = ar" \
	"ARFLAGS = -r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon -lmpi $(MISCLIB) -lm"



##
## Architecture flags for the Intel Paragon XP/S Supercomputer using MPI
## for message passing.  NX is no longer supported by this software.
## Configurations supporting systems with MP-3 nodes are listed, 
## Concurrent I/O is used by default.
##

paragon-thr-mpi:
	$(MAKE) all \
	"ARCH = paragon-thr-mpi" \
	"CC = icc" \
	"CFLAGS = -O4 -Mvect -Mreentrant -Mnoperfmon -Knoieee -I$(MPIINC) -Dparagon -DTHRIO -DTHR $(MISCFLAGS) -DMPI" \
	"AR = ar860" \
	"ARFLAGS = r" \
	"STRIP = strip860" \
	"LIBS = -L. -L$(MPILIB)/paragon/ch_nx -nx -ltachyon -lmpi $(MISCLIB) -lm $(MISCLIB) -lmach3 -lpthreads -lc_r "

paragon-mp-mpi:
	$(MAKE) all \
	"ARCH = paragon-mp-mpi" \
	"CC = icc" \
	"CFLAGS = -O4 -Mvect -Mreentrant -Mnoperfmon -Knoieee -I$(MPIINC) -Dparagon -DTHR $(MISCFLAGS) -DMPI" \
	"AR = ar860" \
	"ARFLAGS = r" \
	"STRIP = strip860" \
	"LIBS = -L. -L$(MPILIB)/paragon/ch_nx -nx -ltachyon -lmpi $(MISCLIB) -lm $(MISCLIB) -lmach3 -lpthreads -lc_r "

paragon-mpi:
	$(MAKE) all \
	"ARCH = paragon-mpi" \
	"CC = icc" \
	"CFLAGS = -O4 -Mvect -Mnoperfmon -Knoieee -I$(MPIINC) -Dparagon $(MISCFLAGS) -DMPI" \
	"AR = ar860" \
	"ARFLAGS = r" \
	"STRIP = strip860" \
	"LIBS = -L. -L$(MPILIB)/paragon/ch_nx -nx -ltachyon -lmpi $(MISCLIB) -lm $(MISCLIB) -lmach3"


##
## Architecture flags for the Intel iPSC/860 Multicomputer using MPI 
## for message passing.  NX is no longer supported by this software. 
## Tested with the mpich distribution from Argonne National Labs
##

ipsc860-mpi:
	$(MAKE) all \
	"ARCH = ipsc860-mpi" \
	"CC = icc" \
	"CFLAGS = -O4 -Mvect -Knoieee -I$(MPIINC) -Dcube $(MISCFLAGS) -DMPI" \
	"AR = ar860" \
	"ARFLAGS = r" \
	"STRIP = strip860" \
	"LIBS = -L. -L$(MPILIB)/intelnx/ch_nx -ltachyon -lmpi $(MISCLIB) -lm"   	

ipsc860-mpi-debug:
	$(MAKE) all \
	"ARCH = ipsc860-mpi-debug" \
	"CC = icc" \
	"CFLAGS = -p -O4 -Mvect -Knoieee -I$(MPIINC) -Dcube $(MISCFLAGS) -DMPI" \
	"AR = ar860" \
	"ARFLAGS = r" \
	"STRIP = touch " \
	"LIBS = -L. -L$(MPILIB)/intelnx/ch_nx -ltachyon -lmpi $(MISCLIB) -lm"   	


##
## OSF1 / Digital Unix 4.0b on Alpha processors
##
tru64-alpha:
	$(MAKE) all \
	"ARCH = tru64-alpha" \
	"CC = cc" \
	"CFLAGS = -std1 -fast -O4 -arch host -tune host -w0 -verbose $(MISCFLAGS) -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -non_shared -om -L. -ltachyon $(MISCLIB) -lm"

##
## OSF1 / Digital Unix 4.0b on Alpha processors with POSIX Threads 
##
tru64-alpha-thr:
	$(MAKE) all \
	"ARCH = tru64-alpha-thr" \
	"CC = cc" \
	"CFLAGS = -speculate all -std1 -pthread -fast -O4 -arch host -tune host -verbose $(MISCFLAGS) -DTHR -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

tru64-alpha-thr-ogl:
	$(MAKE) all \
	"ARCH = tru64-alpha-thr-ogl" \
	"CC = cc" \
	"CFLAGS = -speculate all -std1 -pthread -fast -O4 -arch host -tune host -verbose $(MISCFLAGS) -DTHR -DUSEOPENGL -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(TRU64_GLX_LIBS) $(MISCLIB) -lm"

##
## OSF1 / Digital Unix 4.0d on Alpha processors with Guide for OpenMP
##
tru64-alpha-omp:
	$(MAKE) all \
	"ARCH = tru64-alpha-omp" \
	"CC = guidec" \
	"CFLAGS = -O4 -arch host -tune host $(MISCFLAGS) -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

##
## OSF1 / Digital Unix 4.0d on Alpha processors with Assure for OpenMP
##
## -WApname=<prjfile> needs an absolute path to a project file. This file is
##    created and maintained by assurec.
##
tru64-alpha-ompa:
	$(MAKE) all \
	"ARCH = tru64-alpha-ompa" \
	"CC = assurec" \
	"CFLAGS = -O4 -arch host -tune host $(MISCFLAGS) -WApname=/tacheon/tacheon.prj -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

##
## Architecture flags for Sun Solaris 2.x running on the Sparc platform.
## Expects Sun C 4.2 or newer, and Solaris 2.[3456] or newer.
## Configurations for gcc, and lcc are also listed.
##
solaris:
	$(MAKE) all \
	"ARCH = solaris" \
	"CC = cc" \
	"CFLAGS = -v -xtransition -Xc -fast -xO5 -native -xdepend -fsimple=2 -DSunOS $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

solaris-profile:
	$(MAKE) all \
	"ARCH = solaris-profile" \
	"CC = cc" \
	"CFLAGS = -fast -xO3 -native -g -xpg -DSunOS $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

solaris-profile2:
	$(MAKE) all \
	"ARCH = solaris-profile2" \
	"CC = cc" \
	"CFLAGS = -fast -xO3 -native -g -p -DSunOS $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

solaris-quantify:
	$(MAKE) all \
	"ARCH = solaris-quantify" \
	"CC = quantify cc" \
	"CFLAGS = -native -g -xarch=v8plusa -DSunOS $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch " \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

solaris-purify:
	$(MAKE) all \
	"ARCH = solaris-purify" \
	"CC = purify cc" \
	"CFLAGS = -g -DSunOS $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

solaris-debug:
	$(MAKE) all \
	"ARCH = solaris-debug" \
	"CC = cc" \
	"CFLAGS = -Xc -g -native -DSunOS $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

solaris-lam:
	$(MAKE) all \
	"ARCH = solaris-lam" \
	"CC = hcc" \
	"CFLAGS = -v -xtransition -Xc -fast -xO5 -native -xdepend -fsimple=2 -I$(LAMHOME)/h -DSunOS $(MISCFLAGS) -DMPI" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -L$(LAMHOME)/lib -ltachyon -lmpi $(MISCLIB) -lm -lsocket -lnsl"

solaris-ultra-hpc:
	$(MAKE) all \
	"ARCH = solaris-ultra-hpc" \
	"CC = mpcc" \
	"CFLAGS = -v -xtransition -Xc -fast -xO5 -native -xarch=v8plusa -xdepend -fsimple=2 -DSunOS $(MISCFLAGS) -DMPI" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon -lmpi $(MISCLIB) -lm -lsocket -lnsl" 

solaris-mpi:
	$(MAKE) all \
	"ARCH = solaris-mpi" \
	"CC = cc" \
	"CFLAGS = -v -xtransition -Xc -fast -xO5 -native -xdepend -fsimple=2 -I$(MPIINC) -DSunOS $(MISCFLAGS) -DMPI" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -L$(MPILIB)/solaris/ch_p4 -ltachyon -lmpi $(MISCLIB) -lm -lsocket -lnsl" 

solaris-thr:
	$(MAKE) all \
	"ARCH = solaris-thr" \
	"CC = cc" \
	"CFLAGS = -v -xtransition -Xc -mt -fast -xO5 -native -xdepend -fsimple=2 -DSunOS $(MISCFLAGS) -DTHR" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

solaris-pthreads-gcc:
	$(MAKE) all \
	"ARCH = solaris-pthreads-gcc" \
	"CC = gcc" \
	"CFLAGS = -m486 -Wall -O6 -fomit-frame-pointer -ffast-math -D_REENTRANT -DSunOS $(MISCFLAGS) -DTHR -DUSEPOSIXTHREADS" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lpthread"

solaris-thr-x11:
	$(MAKE) all \
	"ARCH = solaris-thr-x11" \
	"CC = cc" \
	"CFLAGS = -v -xtransition -Xc -mt -fast -xO5 -native -xdepend -fsimple=2 $(X11INC) -DSunOS $(MISCFLAGS) -DTHR -DUSEX11" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm $(X11LIB)"

solaris-v8-thr:
	$(MAKE) all \
	"ARCH = solaris-v8-thr" \
	"CC = cc" \
	"CFLAGS = -v -xtransition -Xc -mt -fast -xO5 -native -xarch=v8 -xdepend -fsimple=2 -DSunOS $(MISCFLAGS) -DTHR  " \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) $(MISCLIB) -lm"

solaris-ultra-thr:
	$(MAKE) all \
	"ARCH = solaris-ultra-thr" \
	"CC = cc" \
	"CFLAGS = -v -xtransition -Xc -mt -fast -xO5 -native -xarch=v8plusa -xdepend -fsimple=2 -DSunOS $(MISCFLAGS) -DTHR  " \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) $(MISCLIB) -lm"

solaris-ultra2-thr:
	$(MAKE) all \
	"ARCH = solaris-ultra-thr-32" \
	"CC = cc" \
	"CFLAGS = -v -Xc -mt -fast -xO5 -native -xtarget=ultra2 -xarch=v8plusa -xdepend -fsimple=2 -DSunOS $(MISCFLAGS) -DTHR  " \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) $(MISCLIB) -lm"

solaris-ultra3-thr:
	$(MAKE) all \
	"ARCH = solaris-ultra3-thr" \
	"CC = cc" \
	"CFLAGS = -v -xtransition -Xc -mt -fast -xO5 -native -xarch=v8plusb -xdepend -fsimple=2 -xalias_level=strong -xprefetch -DSunOS $(MISCFLAGS) -DTHR  " \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) $(MISCLIB) -lm"

solaris-ultra3cu-thr:
	$(MAKE) all \
	"ARCH = solaris-ultra3cu-thr" \
	"CC = /Common/sunopt/SUNWspro.70.3264/bin/cc" \
	"CFLAGS = -v -xtransition -Xc -mt -fast -xO5 -native -xarch=v8plusb -xchip=ultra3cu -xdepend -fsimple=2 -xalias_level=strong -xprefetch -DSunOS $(MISCFLAGS) -DTHR  " \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) $(MISCLIB) -lm"

solaris-64-thr:
	$(MAKE) all \
	"ARCH = solaris-64-thr" \
	"CC = cc" \
	"CFLAGS = -v -Xc -mt -fast -xO5 -native -xarch=v9a -xdepend -fsimple=2 -DSunOS $(MISCFLAGS) -DTHR -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) $(MISCLIB) -lm"

solaris-ultra-thr-ogl:
	$(MAKE) all \
	"ARCH = solaris-ultra-thr-ogl" \
	"CC = cc" \
	"CFLAGS = -v -xtransition -Xc -mt -fast -xO5 -native -xarch=v8plusa -xdepend -fsimple=2 -DSunOS $(MISCFLAGS) -DTHR -DUSEOPENGL  " \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) $(SOLARIS_GLX_LIBS) $(MISCLIB) -lm"

solaris-ultra-hpc-ogl:
	$(MAKE) all \
	"ARCH = solaris-ultra-hpc" \
	"CC = mpcc" \
	"CFLAGS = -v -xtransition -Xc -fast -xO5 -native -xarch=v8plusa -xdepend -fsimple=2 -DSunOS $(MISCFLAGS) -DMPI -DUSEOPENGL" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon -lmpi $(MISCLIB) $(SOLARIS_GLX_LIBS) $(MISCLIB) -lm -lsocket -lnsl" 

solaris-ultra-pthreads-ogl:
	$(MAKE) all \
	"ARCH = solaris-ultra-pthreads-ogl" \
	"CC = cc" \
	"CFLAGS = -v -xtransition -Xc -mt -fast -xO5 -native -xarch=v8plusa -xdepend -fsimple=2 -DSunOS $(MISCFLAGS) -DTHR -DUSEOPENGL -DUSEPOSIXTHREADS  " \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) $(SOLARIS_GLX_LIBS) $(MISCLIB) -lm -lpthread"

solaris-apcc-ultra-thr:
	$(MAKE) all \
	"ARCH = solaris-apcc-ultra-thr" \
	"CC = apcc" \
	"CFLAGS = -fast -O5 -native -Xalnref=0 -Xmopt=4 -Xunroll=1 -Xconstp=2 -Xcopyp=2 -Xflow=1 -Xcg=94 -Xfltacc=2 -Xfltedge=3 -Xintedge=1 -Xfltfold=2 -Xivrep=1 -Xreg=3 -Xsafeintr=1 -Xsched=2 -Xxopt=5 -Xdeflib=2 -Xinllev=5 -XT=ultra2 -DSunOS $(MISCFLAGS) -DTHR -D_REENTRANT  " \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) $(MISCLIB) -lm -lthread"

solaris-gcc-thr:
	$(MAKE) all \
	"ARCH = solaris-gcc-thr" \
	"CC = gcc" \
	"CFLAGS = -ansi -Wall -pedantic -O4 -mv8 -msupersparc -DSunOS $(MISCFLAGS) -DTHR -D_REENTRANT" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lsocket -lthread"

solaris-gcc-thr-x11:
	$(MAKE) all \
	"ARCH = solaris-gcc-thr-x11" \
	"CC = gcc" \
	"CFLAGS = -ansi -Wall -pedantic -O4 -mv8 -msupersparc $(X11INC) -DSunOS $(MISCFLAGS) -DTHR -D_REENTRANT -DUSEX11" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lsocket $(X11LIB) -lthread"

solaris-lcc:
	$(MAKE) all \
	"ARCH = solaris-lcc" \
	"CC = lcc" \
	"CFLAGS = -A -A -DSunOS $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lsocket"

solaris-lcc-thr:
	$(MAKE) all \
	"ARCH = solaris-lcc-thr" \
	"CC = lcc" \
	"CFLAGS = -A -A -DSunOS $(MISCFLAGS) -DTHR -D_REENTRANT" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lsocket -lthread"

solaris-lcc-thr-x11:
	$(MAKE) all \
	"ARCH = solaris-lcc-thr-x11" \
	"CC = lcc" \
	"CFLAGS = -A -A $(X11INC) -DSunOS $(MISCFLAGS) -DTHR -D_REENTRANT -DUSEX11" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lsocket $(X11LIB) -lthread"

##
## SGI Architecture flags, should work with MPI, threads etc.
## Tested with Irix 5.x extensively, Irix 6.x needs retesting
##

irix5-mpi:
	$(MAKE) all \
	"ARCH = irix5-mpi" \
	"CFLAGS = -O2 -mips2 -I$(MPIINC) -DIrix $(MISCFLAGS) -DMPI" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -L$(MPILIB)/IRIX/ch_p4 -ltachyon -lmpi $(MISCLIB) -lm" 

irix5:
	$(MAKE) all \
	"ARCH = irix5" \
	"CFLAGS = -O2 -mips2 -DIrix $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm" 

irix6:
	$(MAKE) all \
	"ARCH = irix6" \
	"CFLAGS = -O3 -mips3 -n32 -DIrix $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm" 

irix6-purify:
	$(MAKE) all \
	"ARCH = irix6-purify" \
	"CC = purify cc" \
	"CFLAGS = -g -O2 -mips3 -n32 -DIrix $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

irix6-64-thr:
	$(MAKE) all \
	"ARCH = irix6-64-thr" \
	"CFLAGS = -woff 1209,1174,3201 -xansi -fullwarn -O3 -mips4 -64 -DIrix -D_REENTRANT -DTHR $(MISCFLAGS) -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lpthread" 

irix6-thr:
	$(MAKE) all \
	"ARCH = irix6-thr" \
	"CFLAGS = -woff 1209,1174,3201 -xansi -fullwarn -O3 -mips3 -n32 -DIrix -D_REENTRANT -DTHR $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lpthread" 

irix6-thr-purify:
	$(MAKE) all \
	"ARCH = irix6-thr-purify" \
	"CC = purify cc" \
	"CFLAGS = -g -O2 -mips3 -n32 -DIrix -D_REENTRANT -DTHR $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lpthread"

irix6-thr-ogl:
	$(MAKE) all \
	"ARCH = irix6-thr-ogl" \
	"CFLAGS = -woff 1209,1174,3201 -xansi -fullwarn -O3 -mips3 -n32 -DIrix -D_REENTRANT -DUSEOPENGL -DTHR $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(IRIX_GLX_LIBS) $(MISCLIB) -lm -lpthread" 


##
## Architecture flags for IBM RS/6000 systems, including the SP-2
## Supercomputer, and PowerPC based RS/6000 systems.  The SP-2 configuration
## requires MPI for message passing.  No multithread configurations are
## available yet, since access to a thread capable test machine is needed
## for implementation.  These configurations require xlc.
##


aix:
	$(MAKE) all \
	"ARCH = aix" \
	"CC = xlc" \
	"CFLAGS = -O3 -DAIX $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

aix-mpi:
	$(MAKE) all \
	"ARCH = aix-mpi" \
	"CC = mpcc" \
	"CFLAGS = -O3 -I$(MPIINC) -DAIX -DMPI $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -us -L. -L$(MPILIB)/rs6000/ch_eui -ltachyon -lmpi $(MISCLIB) -lm"

aix-thr:
	$(MAKE) all \
	"ARCH = aix-thr" \
	"CC = xlc" \
	"CFLAGS = -O3 -DAIX $(MISCFLAGS) -DTHR -D_REENTRANT" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lpthread"

##
## Architecture flags for HP Unix workstations, no MPI configuration
## since no test platform has been available.  The threaded HPUX
## configuration is untested, since no access to a test platform has
## been available.  In theory it should work on a multiprocessor HP
## with POSIX threads capability.
##

hpux:
	$(MAKE) all \
	"ARCH = hpux" \
	"CC = cc" \
	"CFLAGS = -Ae +O4 -DHPUX $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

hpux-thr:
	$(MAKE) all \
	"ARCH = hpux-thr" \
	"CC = cc" \
	"CFLAGS = -Ae +O4 -DHPUX $(MISCFLAGS) -DTHR -D_REENTRANT" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lpthread"

# HP-UX IA-64 using gcc compilers and threads (Merced, Itanium, McKinley, etc)
hpux-ia64-thr:
	$(MAKE) all \
	"ARCH = hpux-ia64-thr" \
	"CC = cc" \
	"CFLAGS = -fast +O3 +Otype_safety=ansi +Onoptrs_to_globals -DHPUX -DTHR -D_REENTRANT $(MISCFLAGS) -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lpthread"


##
## An ancient configuration for NeXT computers, has not been tested
## recently, good luck.
##

next:
	$(MAKE) all \
	"ARCH = next" \
	"CC = cc" \
	"CFLAGS = -O -DNEXT $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

macosx:
	$(MAKE) all \
	"ARCH = macosx" \
	"CC = cc" \
	"CFLAGS = -O4 -fomit-frame-pointer -ffast-math -DBsd $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB)"

macosx-thr:
	$(MAKE) all \
	"ARCH = macosx-thr" \
	"CC = cc" \
	"CFLAGS = -O4 -fomit-frame-pointer -ffast-math -DBsd -DTHR -F/System/Library/Frameworks $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lpthread -framework Carbon"

macosx-altivec:
	$(MAKE) all \
	"ARCH = macosx-altivec" \
	"CC = cc" \
	"CFLAGS = -O4 -mcpu=750 -faltivec -force_cpusubtype_ALL -fomit-frame-pointer -ffast-math -DBsd $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB)"

##
## BeOS
##

beos:
	$(MAKE) all \
	"ARCH = beos" \
	"CC = gcc" \
	"CFLAGS = -m486 -O6 -fomit-frame-pointer -ffast-math $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB)"

##
## An ancient configuration for generic BSD, has not been tested recently,
## good luck.
##

bsd:
	$(MAKE) all \
	"ARCH = bsd" \
	"CC = gcc" \
	"CFLAGS = -O6 -fomit-frame-pointer -ffast-math -DBsd  $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

bsd-sparc:
	$(MAKE) all \
	"ARCH = bsd-sparc" \
	"CC = gcc" \
	"CFLAGS = -mv8 -msupersparc -O6 -fomit-frame-pointer -ffast-math -DBsd  $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

##
## Windows (Win32) Builds using Cygnus' CygWin32 libraries and egcs
##

win32:
	$(MAKE) all \
	"ARCH = win32" \
	"CC = gcc" \
	"CFLAGS = -mpentium -Wall -O6 -fomit-frame-pointer -ffast-math -DWIN32 $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = echo" \
	"RANLIB = echo" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

win32-mpi:
	$(MAKE) all \
	"ARCH = win32-mpi" \
	"CC = gcc" \
	"CFLAGS = -mpentium -Wall -O6 -fomit-frame-pointer -ffast-math -DWIN32 $(MISCFLAGS) -I'/Program files/MPIPro/INCLUDE' -DMPI" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = echo" \
	"RANLIB = echo" \
	"LIBS = -L. -L'/Program files/MPIPro/LIB' -ltachyon -lmpi $(MISCLIB) -lm"

##
## Configurations of Tachyon for various versions of Linux
##

# Linux on x86, using gcc
linux:
	$(MAKE) all \
	"ARCH = linux" \
	"CC = gcc" \
	"CFLAGS = -m486 -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux x86, on AMD Athlon, using gcc
linux-intelcc:
	$(MAKE) all \
	"ARCH = linux-intelcc" \
	"CC = icc" \
	"CFLAGS = -march=pentium4 -ansi -align -fno-alias -O3 -rcd -vec_report3 -ip -unroll -static -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"
#
#	"CFLAGS = -ansi -align -fno-alias -O3 -rcd -tpp7 -vec_report3 -xiMKW -ipo -unroll -static -DLinux $(MISCFLAGS)" \
#	"CFLAGS = -ansi -restrict -O3 -rcd -tpp7 -vec_report3 -xiMKW -ip -unroll -static -DLinux $(MISCFLAGS)" \
#

# Linux x86, on AMD Athlon, using gcc
linux-p4:
	$(MAKE) all \
	"ARCH = linux-p4" \
	"CC = gcc" \
	"CFLAGS = -mcpu=i686 -march=i686 -funroll-loops -fexpensive-optimizations -malign-double -fschedule-insns2 -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux x86, on AMD Athlon, using gcc
linux-athlon:
	$(MAKE) all \
	"ARCH = linux-athlon" \
	"CC = /usr/bin/gcc" \
	"CFLAGS = -mcpu=athlon -march=athlon -funroll-loops -fexpensive-optimizations -malign-double -fschedule-insns2 -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux x86/Pentium with Portland group compilers
linux-pgcc:
	$(MAKE) all \
	"ARCH = linux-pgcc" \
	"CC = pgcc" \
	"CFLAGS = -fast -Mvect=assoc,prefetch -tp p6 -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux x86/athlon with Portland group compilers
linux-athlon-pgcc:
	$(MAKE) all \
	"ARCH = linux-athlon-pgcc" \
	"CC = pgcc" \
	"CFLAGS = -fast -Mvect=assoc,prefetch -tp athlon -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux x86 using gcc, with threads
linux-thr:
	$(MAKE) all \
	"ARCH = linux-thr" \
	"CC = gcc" \
	"CFLAGS = -m486 -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux -DTHR -D_REENTRANT $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lpthread"

# Linux x86 using gcc, threads, and OpenGL
linux-thr-ogl:
	$(MAKE) all \
	"ARCH = linux-thr-ogl" \
	"CC = cc" \
	"CFLAGS = -m486 -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux -DTHR -D_REENTRANT $(MISCFLAGS) -DUSEOPENGL $(LINUX_GLX_INCS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = touch" \
	"LIBS = -L. -ltachyon $(MISCLIB) $(LINUX_GLX_LIBS) $(MISCLIB) -lm -lpthread"

# Linux x86 using Scyld's beowulf distribution and OpenGL
linux-beowulf-mpi-ogl:
	$(MAKE) all \
	"ARCH = linux-beowulf-mpi" \
	"CC = gcc" \
	"CFLAGS = -m486 -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux -DMPI $(MISCFLAGS) -DUSEOPENGL $(LINUX_GLX_INCS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon -lmpi $(MISCLIB) $(LINUX_GLX_LIBS) $(MISCLIB) -lm"

# Linux x86 using Scyld's beowulf distribution
linux-beowulf-mpi:
	$(MAKE) all \
	"ARCH = linux-beowulf-mpi" \
	"CC = gcc" \
	"CFLAGS = -m486 -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux -DMPI $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon -lmpi $(MISCLIB) -lm"

# Linux x86 using LAM MPI
linux-lam:
	$(MAKE) all \
	"ARCH = linux-lam" \
	"CC = hcc" \
	"CFLAGS = -m486 -Wall -O6 -fomit-frame-pointer -ffast-math -I$(LAMHOME)/h -DLinux -DMPI $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -L$(LAMHOME)/lib -ltachyon -lmpi $(MISCLIB) -lm"

# Linux x86 using LAM MPI and threads
linux-lam-thr:
	$(MAKE) all \
	"ARCH = linux-lam-thr" \
	"CC = hcc" \
	"CFLAGS = -m486 -Wall -O6 -fomit-frame-pointer -ffast-math -I$(LAMHOME)/h -DLinux -DMPI -DTHR -D_REENTRANT $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -L$(LAMHOME)/lib -ltachyon -lmpi $(MISCLIB) -lm -lpthread"

# Linux on iPaq, using gcc
linux-ipaq:
	$(MAKE) all \
	"ARCH = linux-ipaq" \
	"CC = gcc" \
	"CFLAGS = -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux PPC using gcc 
linux-ppc:
	$(MAKE) all \
	"ARCH = linux-ppc" \
	"CC = gcc" \
	"CFLAGS = -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux PPC using gcc 
linux-ps2:
	$(MAKE) all \
	"ARCH = linux-ps2" \
	"CC = gcc" \
	"CFLAGS = -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux Alpha using gcc 
linux-alpha:
	$(MAKE) all \
	"ARCH = linux-alpha" \
	"CC = gcc" \
	"CFLAGS = -Wall -O6 -fomit-frame-pointer -ffast-math -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux Alpha using Compaq's compilers
linux-alpha-ccc:
	$(MAKE) all \
	"ARCH = linux-alpha-ccc" \
	"CC = ccc" \
	"CFLAGS = -std1 -fast -O4 -arch host -tune host -w0 -verbose -DLinux $(MISCFLAGS)" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux Alpha using Compaq's compilers, and the Quadrics MPI implementation
linux-alpha-ccc-qsw:
	$(MAKE) all \
	"ARCH = linux-alpha-ccc" \
	"CC = ccc" \
	"CFLAGS = -std1 -fast -O4 -arch host -tune host -w0 -verbose -DLinux -DMPI $(MISCFLAGS) -I/usr/lib/mpi/include" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lelan -lmpi -L/usr/lib/mpi/lib"

# Linux IA-64 using gcc (Merced, Itanium, McKinley, etc)
linux-ia64:
	$(MAKE) all \
	"ARCH = linux-ia64" \
	"CC = cc" \
	"CFLAGS = -O3 -DLinux $(MISCFLAGS) -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux IA-64 using SGI compilers (Merced, Itanium, McKinley, etc)
linux-ia64-sgicc:
	$(MAKE) all \
	"ARCH = linux-ia64-sgicc" \
	"CC = sgicc" \
	"CFLAGS = -O3 -DLinux $(MISCFLAGS) -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm"

# Linux IA-64 using gcc compilers and threads (Merced, Itanium, McKinley, etc)
linux-ia64-thr:
	$(MAKE) all \
	"ARCH = linux-ia64-thr" \
	"CC = cc" \
	"CFLAGS = -O3 -DLinux -DTHR -D_REENTRANT $(MISCFLAGS) -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lpthread"

# Linux IA-64 using SGI compilers and threads (Merced, Itanium, McKinley, etc)
linux-ia64-thr-sgicc:
	$(MAKE) all \
	"ARCH = linux-ia64-thr-sgicc" \
	"CC = sgicc" \
	"CFLAGS = -O3 -DLinux -DTHR -D_REENTRANT $(MISCFLAGS) -DLP64" \
	"AR = ar" \
	"ARFLAGS = r" \
	"STRIP = strip" \
	"RANLIB = ranlib" \
	"LIBS = -L. -ltachyon $(MISCLIB) -lm -lpthread"

##
## CSPI PowerPC Based Multicomputers Running VXWORKS
## This configuration works for the machine at MPI Software Technologies
## Uses MSTI MPI/Pro for message passing.
## 
cspi-ppc-mpi:
	$(MAKE) all \
	"ARCH = cspi-ppc-mpi" \
	"CC = ccppc" \
	"CFLAGS = -O3 -fomit-frame-pointer -ffast-math -DCPU=PPC603 -DVXWORKS -I$(WIND_BASE)/target/h -I/home/projects/CSPI/mpi_pro/include -DMPI $(MISCFLAGS)" \
	"AR = arppc" \
	"ARFLAGS = -r" \
	"STRIP = /bin/touch" \
	"LIBS = -nostartfiles -Wl,-r -L. -L/home/projects/CSPI/mpi_pro/lib/ppc603/bdmp -ltachyon -lmpi"


##
## Mercury PowerPC Based Multicomputers Running MCOS
## This configuration works for the machine at MPI Software Technologies
## Uses MSTI MPI/Pro for message passing.
## 
mercury-ppc-mpi:
	$(MAKE) all \
	"ARCH = mercury-ppc-mpi" \
	"CC = ccmc -t ppc -x " \
	"CFLAGS = -w0 -Ot -to ppc:c:-O3,-fomit-frame-pointer,-ffast-math -I/opt/MPIPro/include -DMPI -DMCOS $(MISCFLAGS)" \
	"AR = armc" \
	"ARFLAGS = -r" \
	"STRIP = /bin/touch" \
	"LIBS = -L. -L/opt/MPIPro/lib ../compile/mercury-ppc-mpi/libmgf.a ../compile/mercury-ppc-mpi/libray.a -lmpi.appc"

	(cd ../librtvi; $(MAKE););
	$(MAKE) all \
	"ARCH = mercury-ppc-mpi-rtvi" \
	"CC = ccmc -t ppc -x " \
	"CFLAGS = -w0 -Ot -to ppc:c:-O3,-fomit-frame-pointer,-ffast-math -I/opt/MPIPro/include -DMPI -DMCOS -DUSESINGLEFLT -DUSERTVI $(RTVIINC)" \
	"AR = armc" \
	"ARFLAGS = -r" \
	"STRIP = /bin/touch" \
	"LIBS = -L. -L/opt/MPIPro/lib ../compile/mercury-ppc-mpi-rtvi/libmgf.a ../compile/mercury-ppc-mpi-rtvi/libray.a $(RTVILIB) -lmpi.appc"
	@echo "Note: Remember to link ray.ppc -> ray before your first run."	
	@echo "      Also, copy your machines file into your CWD."

##
## Mercury i860 Based Multicomputers Running MCOS
## Single CPU build, with RTVI video output.
##
mercury-i860-rtvi:
	$(MAKE) all \
	"ARCH = mercury-i860-rtvi" \
	"CC = ccmc -t i860 -x " \
	"CFLAGS = -w0 -Ot -DMCOS $(MISCFLAGS)" \
	"AR = armc" \
	"ARFLAGS = -r" \
	"STRIP = /bin/touch" \
	"LIBS = -L. ../compile/mercury-i860-rtvi/libmgf.a ../compile/mercury-i860-rtvi/libray.a"

##
## Mercury i860 Based Multicomputers Running MCOS
## This configuration works for the machine at MPI Software Technologies
## Uses MSTI MPI/Pro for message passing.
## 
mercury-i860-mpi:
	$(MAKE) all \
	"ARCH = mercury-i860-mpi" \
	"CC = ccmc -t i860 -x " \
	"CFLAGS = -w0 -Ot -I/opt/MPIPro/include -DMPI -DMCOS $(MISCFLAGS)" \
	"AR = armc" \
	"ARFLAGS = -r" \
	"STRIP = /bin/touch" \
	"LIBS = -L. -L/opt/MPIPro/lib ../compile/mercury-i860-mpi/libmgf.a ../compile/mercury-i860-mpi/libray.a -lmpi"

